<?php

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Http;
use App\Jobs\SendOrderNotification;
use App\Models\Order;
use App\Models\OrderPayment;
use App\Models\OrderDetails;
use App\Models\DeliveryCharge;
use App\Utils\ModulUtil;
use App\Utils\Util;
use App\Models\CouponCode;
use App\Models\User;
use App\Models\Product;
use App\Models\Variation;
use App\Models\Information;
use App\Facades\FacebookConversion;
use Illuminate\Support\Facades\Schema; // ✅ Added Schema Support

class CheckoutController extends Controller
{
    public $modulutil;
    public $util;

    public function __construct(ModulUtil $modulutil, Util $util){
        $this->util=$util;
        $this->modulutil=$modulutil;
    }

    /** ===============================
     * ✅ NEW: WORKER ASSIGNMENT LOGIC
     * =============================== */
    private function getActiveWorkerIds()
    {
        return User::query()
            ->where(function ($q) {
                $q->whereNull('status')
                    ->orWhereIn('status', [1, '1', true, 'true', 'active', 'Active']);
            })
            ->when(
                Schema::hasColumn((new User)->getTable(), 'deleted_at'),
                fn($q) => $q->whereNull('deleted_at')
            )
            ->whereHas('roles', fn($q) => $q->whereRaw('LOWER(name)=?', ['worker']))
            ->orderBy('id')
            ->pluck('id');
    }

    private function pickNextWorkerId()
    {
        $activeIds = $this->getActiveWorkerIds();
        
        // যদি কোনো ওয়ার্কার না থাকে, তবে ডিফল্ট হিসেবে এডমিন (ID 1) কে দেওয়া হবে
        if ($activeIds->isEmpty()) {
            return 1;
        }

        $candidateId = DB::table('users as u')
            ->join('model_has_roles as m', 'm.model_id', '=', 'u.id')
            ->join('roles as r', 'r.id', '=', 'm.role_id')
            ->leftJoin('orders as o', function ($join) {
                $join->on('o.assign_user_id', '=', 'u.id')
                    ->whereDate('o.created_at', DB::raw('CURDATE()'));
            })
            ->whereIn('u.id', $activeIds->toArray())
            ->whereRaw('LOWER(r.name) = ?', ['worker'])
            ->where(function ($q) {
                $q->whereNull('u.status')
                    ->orWhereIn('u.status', [1, '1', true, 'true', 'active', 'Active']);
            })
            ->groupBy('u.id')
            ->orderByRaw('COUNT(o.id) ASC') // যার কাজ সবচেয়ে কম তাকে আগে দেওয়া হবে
            ->orderBy('u.id', 'ASC')
            ->value('u.id');

        return (int) ($candidateId ?? $activeIds->first());
    }
    
    // index, courierPercentage, callApi... (No Changes)
    public function index(){
        $cart = session()->get('cart', []);
        if (empty($cart)) { return redirect()->route('front.home'); }
        $charges = DeliveryCharge::whereNotNull('status')->get();
        $coupon        = session()->get('coupon_discount');
        $coupn_item = CouponCode::where('amount', $coupon)->first();
        $cart  = session()->get('cart');
        $total = getCouponDiscount();
        
        try {
            $eventId = "IC_" . now()->format('Ymdhi');
            $contents   = [];
            $contentIds = [];
            $totalValue = 0;
            foreach ($cart as $item) {
                $contents[] = [
                    'id'             => $item['product_id'],
                    'quantity'       => $item['quantity'],
                    'item_price'     => $item['price']
                ];
                $contentIds[] = $item['product_id'];
                $totalValue  += $item['price'] * $item['quantity'];
            }
            FacebookConversion::sendEvent('InitiateCheckout', [
                'currency'      => 'BDT',
                'value'         => $totalValue,
                'content_ids'   => $contentIds,
                'contents'      => $contents,
                'num_items'     => count($cart),
                'content_type'  => 'product'
            ], $eventId);
        } catch (\Exception $e) { \Log::error('Facebook CAPI BeginCheckout Error: ' . $e->getMessage()); }

        $totalPrice = 0;
        foreach ($cart as $item) { $totalPrice += $item['price'] * $item['quantity']; }
        if (($coupn_item) && ($coupon > 0) && ($coupn_item->minimum_amount > $total)) {
            session()->put('coupon_discount',null);
            session()->put('discount_type',null);
        }
        return view('frontend.cart.checkout', compact('cart','charges','totalPrice'));
    }

    public function courierPercentage(Request $request){
        $id     = $request->id;
        $number = $request->phone;
        if($id){
            $customer = User::findOrFail($id);
            if($number){
                $checkCourier = $this->callApi($number);
                if(isset($checkCourier)){
                    $customer->curier_summery = $checkCourier;
                    $customer->save();
                }
            }
        }
    }
    
    private function callApi($number){
        $info   = Information::first();
        $apiKey = $info->fraudApi;
        $url    = "https://dash.hoorin.com/api/courier/sheet.php?apiKey=$apiKey&searchTerm=$number";
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $response = curl_exec($ch);
        curl_close($ch);
        return $response;
    }

    // ✅ storelandData (Landing Page Order)
    public function storelandData(Request $request) {
        $data = $request->validate([
            'mobile'             => 'required|digits_between:11,11',
            'first_name'         => 'required',
            'payment_method'     => '',
            'shipping_address'   => 'required',
            'note'               => '',
            'delivery_charge_id' => 'required|numeric',
            'final_amount'       => '',
            'amount'             => '',
            'purchase_event_id'  => 'nullable|string',
            'prd_id'             => 'required', 
            'variation_id'       => 'nullable',
            'quantity'           => 'nullable'
        ]);

        $info = Information::first();
        if(isset($info->otp_system) && $info->otp_system == 1) {
            if(!session()->has('otp_verified') || session()->get('otp_verified') !== true) {
                 return response()->json([
                    'success' => false, 
                    'msg' => 'অর্ডার কনফার্ম করার জন্য মোবাইল ভেরিফিকেশন প্রয়োজন।'
                ]);
            }
        }
        
        $product = Product::with('variations')->where('id', $request->prd_id)->first();
        if(!$product) {
            return response()->json(['success' => false, 'msg' => 'Product not found!']);
        }

        $quantity = $request->quantity;
        $proQty   = ($quantity == null || $quantity == '') ? 1 : (int)$quantity;
        $unit_price = $request->amount ?? $product->sell_price;
        $subTotal = $unit_price * $proQty;

        // ✅✅✅ NEW: Validate Max Quantity and Max Amount from Settings (Landing Page)
        if (isset($info->max_order_qty) && $info->max_order_qty > 0) {
            if ($proQty > $info->max_order_qty) {
                return response()->json(['success' => false, 'msg' => "You can order a maximum of {$info->max_order_qty} items at a time."]);
            }
        }
        if (isset($info->max_order_amount) && $info->max_order_amount > 0) {
            if ($subTotal > $info->max_order_amount) {
                return response()->json(['success' => false, 'msg' => "Your order amount cannot exceed ৳{$info->max_order_amount}"]);
            }
        }

        if (empty(auth()->user()->id)) {
            $user = User::where('mobile', $request->mobile)->first();
            if(!$user) {
                $user = User::create([
                    'first_name'       => $request->first_name,
                    'mobile'           => $request->mobile,
                    'shipping_address' => $request->shipping_address,
                    'note'             => $request->note,
                    'username'         => strtolower(str_replace(' ', '', $request->first_name)) . rand(100,999),
                    'status'           => 1
                ]);
            } else {
                $user->update([
                    'first_name' => $request->first_name,
                    'shipping_address' => $request->shipping_address
                ]);
            }
            $data['user_id'] = $user->id;
        } else {
            $user = auth()->user(); 
            $data['user_id'] = $user->id;
        }

        $total_discount_val = $proQty * ($product['discount'] ?? 0);
        
        $pr_data = [
            'product_id'     => $request->prd_id,
            'quantity'       => $proQty,
            'unit_price'     => $unit_price,
            'discount'       => $product['discount'] ?? 0,
            'is_stock'       => $product['is_stock'] ?? 1,
            'purchase_price' => $product['purchase_prices'] ?? 0,
            'variation_id'   => $request['variation_id']
        ];
        
        $charge = DeliveryCharge::where('id', $data['delivery_charge_id'])->first();
        $chargeAmount = $charge ? $charge->amount : 0;
        
        $data['date'] = date('Y-m-d');
        
        // ✅✅✅ Assign to Worker (Load Balancing)
        $data['assign_user_id'] = $this->pickNextWorkerId(); 
        
        $data['invoice_no']      = rand(111111,999999);
        $data['discount']        = $total_discount_val;
        $data['shipping_charge'] = $chargeAmount;
        $data['courier_id']      = 3; 

        $coupon_discount = session()->get('coupon_discount') ?? 0;
        
        $data['amount'] = $subTotal; 
        $data['discount'] = $total_discount_val + $coupon_discount; 
        $data['final_amount'] = ($subTotal + $chargeAmount) - $coupon_discount; 
        
        unset($data['payment_method']);
        unset($data['purchase_event_id']);
        unset($data['prd_id']);
        unset($data['variation_id']);
        unset($data['quantity']);

        DB::beginTransaction();
        try {
            $order = Order::where('status', 'incomplete')
                ->where(function($query) use ($user, $request) {
                    if ($user) $query->where('user_id', $user->id);
                    if (!empty($request->mobile)) $query->orWhere('mobile', $request->mobile);
                })
                ->latest()
                ->lockForUpdate()
                ->first();

            if($order){
                $data['status'] = 'pending';
                // ইনকমপ্লিট অর্ডার আপডেট হলে আবার নতুন ওয়ার্কার অ্যাসাইন করার দরকার নেই যদি আগে থাকে,
                // তবে ইউজার চাইলে এখানেও ফোর্স অ্যাসাইন করতে পারে। আমরা এখানে নতুন অ্যাসাইন রাখছি।
                $order->update($data);
                DB::table('order_details')->where('order_id', $order->id)->delete();
                if (!empty($pr_data)) {
                    $order->details()->create($pr_data);
                }
            } else {
                $data['status'] = 'pending';
                $order = Order::create($data);
                if (!empty($pr_data)) {
                    $order->details()->create($pr_data);
                }
            }

            // Stock
            if($product->stock_quantity >= $proQty) {
                $product->decrement('stock_quantity', $proQty);
                $this->checkAndSendStockAlert($product);
            }

            $this->modulutil->orderPayment($order, $request->all());
            $this->modulutil->orderstatus($order);
            $this->sendAdminNotification($order);

            session()->forget(['coupon_discount', 'otp_verified']);

            $url = route('front.confirmOrderlanding',[$order->id]);
            session()->put('cart',[]);
            session()->put('coupon_discount',null);
            session()->put('discount_type',null);

            DB::commit();
            
            if($request->ajax()){
                return response()->json([
                    'success' => true,
                    'msg'     => 'Checkout Successfully..!!',
                    'url'     => $url,
                    'purchase_event_id' => $request->purchase_event_id
                ]);
            } else {
                return redirect($url);
            }

        } catch (\Exception $e) {
            DB::rollback();
            if($request->ajax()){
                return response()->json(['success'=>false,'msg'=>$e->getMessage()]);
            } else {
                return back()->with('error', $e->getMessage());
            }
        }
    }
    
    // ✅ incompleteStore
    public function incompleteStore(Request $request){
        $req_data = $request->validate([
            'mobile'       => 'required|numeric|min:11',
            'name'         => 'nullable',
            'address'      => 'nullable',
            'prd_id'       => 'nullable',
            'amount'       => 'nullable',
            'quantity'     => 'nullable',
            'variation_id' => 'nullable'
        ]);
        
        DB::beginTransaction();
        try {
            $existingOrder = Order::where('mobile', $req_data['mobile'])
                ->where('status', 'incomplete')
                ->latest()
                ->lockForUpdate()
                ->first();

            $user = null;
            if (!empty($request->mobile)) {
                 $user = User::where('mobile', $request->mobile)->first();
                 if(!$user) {
                     $user = User::create([
                         'mobile'           => $request->mobile,
                         'first_name'       => $req_data['name'] ?? 'Guest',
                         'username'         => strtolower(str_replace(' ', '', $req_data['name'] ?? 'guest')) . rand(100,999),
                         'status'           => 1,
                         'shipping_address' => $req_data['address'] ?? ''
                     ]);
                 }
            }
            
            $product_list = [];
            $unique_check = []; 
            $total = 0;
            $total_discount = 0;
            $coupn_discount = getCouponDiscount() ?? 0;

            if($request->has('prd_id') && !empty($request->prd_id)) {
                $prodInfo = Product::find($request->prd_id);
                if($prodInfo) {
                    $qty = $request->quantity ?? 1;
                    $price = $request->amount ?? $prodInfo->sell_price;
                    $total = $price * $qty;
                    $product_list[] = [
                        'product_id'     => $request->prd_id,
                        'quantity'       => $qty,
                        'unit_price'     => $price,
                        'purchase_price' => $prodInfo->purchase_price,
                        'variation_id'   => $request->variation_id ?? null,
                        'discount'       => 0,
                        'is_stock'       => $prodInfo->is_stock,
                    ];
                }
            } else {
                $carts = session()->get('cart',[]);
                if ($carts) {
                    foreach($carts as $key=>$item){
                        $total          += $item['quantity'] * $item['price'];
                        $total_discount += $item['quantity'] * ($item['discount'] ?? 0);
                        
                        $uniqueKey = $item['product_id'] . '_' . ($item['variation_id'] ?? 0);

                        if(isset($unique_check[$uniqueKey])) {
                            $product_list[$unique_check[$uniqueKey]]['quantity'] += $item['quantity'];
                        } else {
                            $product_list[] = [
                                'product_id'     => $item['product_id'],
                                'quantity'       => $item['quantity'],
                                'unit_price'     => $item['price'],
                                'purchase_price' => $item['purchase_price'] ?? 0,
                                'variation_id'   => $item['variation_id'],
                                'discount'       => $item['discount'] ?? 0,
                                'is_stock'       => $item['is_stock'] ?? 1,
                            ];
                            $unique_check[$uniqueKey] = count($product_list) - 1;
                        }
                    }
                } 
            }
            
            if(empty($product_list)) {
                DB::rollback();
                return response()->json(['success' => false, 'message'=>'No products to save']);
            }

            $data = [
                'date'             => date('Y-m-d'),
                'invoice_no'       => rand(111111,999999),
                'discount'         => $total_discount + $coupn_discount,
                'amount'           => $total_discount + $total,
                'shipping_charge'  => 0, 
                'first_name'       => $req_data['name'] ?? ($user->first_name ?? ''),
                'mobile'           => $req_data['mobile'],
                'shipping_address' => $req_data['address'] ?? '',
                'status'           => 'incomplete',
                'final_amount'     => $total - $coupn_discount,
                'user_id'          => $user ? $user->id : null,
                
                // ✅✅✅ Assign to Worker (Load Balancing) - Even for Incomplete Orders
                'assign_user_id'   => $this->pickNextWorkerId() 
            ];

            if ($existingOrder) {
                $existingOrder->update($data);
                DB::table('order_details')->where('order_id', $existingOrder->id)->delete();
                $existingOrder->details()->createMany($product_list);
            } else {
                $order = Order::create($data);
                $order->details()->createMany($product_list);
            }
            
            DB::commit();
            return response()->json(['success' => true, 'message'=>'Incomplete Order Saved']);

        } catch (\Exception $e) {
            DB::rollback();
            return response()->json(['success' => false, 'message' => $e->getMessage()]);
        }
    }

    // ✅ store (Final Order)
    public function store(Request $request){
        $data = $request->validate([
            'mobile'             => 'required|digits_between:11,11',
            'first_name'         => 'required',            
            'payment_method'     => 'required',
            'shipping_address'   => 'required',        
            'ip_address'         => '',
            'note'               => '',
            'delivery_charge_id' => 'required|numeric',              
        ]);

        $user = auth()->user(); 
        if($request->ip_address == null){
            $data['ip_address'] = $request->ip();
        }

        // Fraud Check
        $info         = Information::first();
        $limitMinutes = $info->time_limit ?? 60;
        
        $appliesMobileCheck = ($info->is_mobile_check == 1) && !empty($request->mobile);
        $appliesIpCheck     = ($info->is_ip_check == 1) && !empty($data['ip_address']);
        
        if ($appliesMobileCheck || $appliesIpCheck) {
            $query = Order::whereNot('status', 'incomplete');
            $query->where(function($q) use ($appliesMobileCheck, $appliesIpCheck, $request, $data) {
                if ($appliesMobileCheck) $q->where('mobile', $request->mobile);
                if ($appliesIpCheck) $q->orWhere('ip_address', $data['ip_address']);
            });
            $recentOrder = $query->where('created_at', '>=', now()->subMinutes($limitMinutes))->latest()->first();
            if ($recentOrder) {
                $minutesPassed = now()->diffInMinutes($recentOrder->created_at);
                $remaining     = max(0, $limitMinutes - $minutesPassed);
                return response()->json([
                    'success' => false,
                    'msg'     => "You can place a new order after {$remaining} minutes."
                ]);
            }
        }

        $carts          = session()->get('cart',[]);
        $coupn_discount = getCouponDiscount() ?? 0;
        
        $total_cart_qty = 0;
        $total          = 0;

        if ($carts) {
            foreach($carts as $item){
                $total_cart_qty += $item['quantity'];
                $total          += $item['quantity'] * $item['price'];
            }
        }

        // ✅✅✅ NEW: Validate Max Quantity and Max Amount from Settings (Main Website)
        if (isset($info->max_order_qty) && $info->max_order_qty > 0) {
            if ($total_cart_qty > $info->max_order_qty) {
                if($request->ajax()) {
                    return response()->json(['success' => false, 'msg' => "You can order a maximum of {$info->max_order_qty} items at a time."]);
                }
                return back()->with('error', "You can order a maximum of {$info->max_order_qty} items at a time.");
            }
        }

        if (isset($info->max_order_amount) && $info->max_order_amount > 0) {
            if ($total > $info->max_order_amount) {
                if($request->ajax()) {
                    return response()->json(['success' => false, 'msg' => "Your order amount cannot exceed ৳{$info->max_order_amount}"]);
                }
                return back()->with('error', "Your order amount cannot exceed ৳{$info->max_order_amount}");
            }
        }

        if (!empty($request->mobile)) {
            $baseUsername = strtolower(str_replace(' ', '', $data['first_name']));
            $username     = $baseUsername;
            $counter      = 1;
            while (User::where('username', $username)->exists()) {
                $username = $baseUsername . $counter;
                $counter++;
            }
            $userUpdated = User::updateOrCreate(
                ['mobile' => $request->mobile],
                [
                    'first_name' => $data['first_name'],
                    'username'   => $username,
                    'status'     => 1,
                ]
            );
            $data['user_id'] = $userUpdated->id;
            $user = $userUpdated;
        }

        if (auth()->check()) {
            $data['user_id'] = auth()->id();
            $user = auth()->user();
        }

        $product_list   = [];
        $unique_check   = []; 
        $total_discount = 0;

        if ($carts) {
            foreach($carts as $key=>$item){
                $total_discount += $item['quantity'] * ($item['discount'] ?? 0);
                
                $uniqueKey = $item['product_id'] . '_' . ($item['variation_id'] ?? 0);

                if(isset($unique_check[$uniqueKey])) {
                    $product_list[$unique_check[$uniqueKey]]['quantity'] += $item['quantity'];
                } else {
                    $product_list[] = [
                        'product_id'     => $item['product_id'],
                        'quantity'       => $item['quantity'],
                        'unit_price'     => $item['price'],
                        'variation_id'   => $item['variation_id'],
                        'purchase_price' => $item['purchase_price'] ?? 0,
                        'discount'       => $item['discount'] ?? 0,
                        'is_stock'       => $item['is_stock'] ?? 1,
                    ];
                    $unique_check[$uniqueKey] = count($product_list) - 1;
                }
            }
        } 

        $charge = DeliveryCharge::find($data['delivery_charge_id']);
        $chargeAmount = $charge ? $charge->amount : 0;
        
        $data['date'] = date('Y-m-d');
        
        // ✅✅✅ Assign to Worker (Load Balancing)
        $data['assign_user_id'] = $this->pickNextWorkerId(); 

        $data['invoice_no']      = rand(111111,999999);
        $data['discount']        = $total_discount + $coupn_discount;
        $data['amount']          = $total_discount + $total;
        $data['shipping_charge'] = $chargeAmount;
        $data['final_amount']    = $total + $chargeAmount - $coupn_discount;        
        
        DB::beginTransaction();
        try {
            unset($data['payment_method']);

            $order = Order::where('status', 'incomplete')
                ->where(function($query) use ($user, $request) {
                    if ($user) $query->where('user_id', $user->id);
                    if (!empty($request->mobile)) $query->orWhere('mobile', $request->mobile);
                })
                ->latest()
                ->lockForUpdate()
                ->first();

            if(!$order){
                $data['status'] = 'pending';
                $order = Order::create($data);
                if (!empty($product_list)) { 
                    foreach ($product_list as $item) {
                        $pro = Product::find($item['product_id']);
                        if($pro->stock_quantity < $item['quantity']){
                            DB::rollback();
                            if($request->ajax()) return response()->json(['success'=>false,'msg'=>'Stock Not Available!']);
                            return back()->with('error', 'Stock Not Available!');
                        } else {
                            $pro->decrement('stock_quantity', $item['quantity']);
                            $this->checkAndSendStockAlert($pro);
                        }
                    }   
                    $order->details()->createMany($product_list);
                }   
            } else {  
                DB::table('order_details')->where('order_id', $order->id)->delete();
                $order->details()->createMany($product_list);

                foreach ($product_list as $item) {
                    $pro = Product::find($item['product_id']);
                    if($pro->stock_quantity < $item['quantity']){
                        DB::rollback();
                        if($request->ajax()) return response()->json(['success'=>false,'msg'=>'Stock Not Available!']);
                        return back()->with('error', 'Stock Not Available!');
                    } else {
                         $pro->decrement('stock_quantity', $item['quantity']);
                         $this->checkAndSendStockAlert($pro);
                    }
                }

                $data['status'] = 'pending';
                $order->update($data);
            }
            
            $this->modulutil->orderPayment($order, $request->all());
            $this->modulutil->orderstatus($order);
            $this->sendAdminNotification($order);

            DB::commit();        
            
            try {
                $eventId = "PUR_" . $order->id;
                $contents   = [];
                $contentIds = [];
                foreach ($order->details as $sellProduct) {
                    $contents[] = [
                        'id'             => $sellProduct->product_id,
                        'quantity'       => $sellProduct->quantity,
                        'item_price'     => $sellProduct->unit_price
                    ];
                    $contentIds[] = $sellProduct->product_id;
                }
                
                FacebookConversion::sendPurchase([
                    'currency'      => 'BDT', 
                    'value'         => $order->final_amount,
                    'content_ids'   => $contentIds,
                    'contents'      => $contents,
                    'order_id'      => $order->id,
                    'num_items'     => $order->details()->sum('quantity'),
                ], $eventId);
            } catch (\Exception $e) { \Log::error('Facebook CAPI Purchase Error: ' . $e->getMessage()); }
            
            session()->put('cart',[]);
            session()->put('coupon_discount',null);
            session()->put('discount_type',null); 
            
            $url = route('front.confirmOrder',[$order->id]);
            
            if ($request->ajax()) {
                return response()->json([
                    'success' => true,
                    'msg'     => 'Order Create successfully!',
                    'url'     => $url
                ]);
            } else {
                return redirect($url);
            }

        } catch (\Exception $e) {
            DB::rollback();
            if ($request->ajax()) {
                return response()->json(['success' => false, 'msg' => $e->getMessage()]);
            } else {
                return back()->with('error', $e->getMessage());
            }
        }
    }
    
    public function storeData(Request $request) { return $this->store($request); }
    public function StoreChk(Request $request){
          $this->validate($request, [
            'first_name'         => 'required',
            'mobile'             => 'required',
            'shipping_address'   => 'required',
            'delivery_charge_id' => 'required'
        ]);
    }
    
    public function getCouponDiscount(Request $request){
        $info = Information::first();
        if(isset($info->coupon_visibility) && $info->coupon_visibility == 0){
             return response()->json(['success'=>false, 'msg' => 'Coupon system is currently disabled.']);
        }
        $data = $request->validate([ 'code' => 'required' ]);
        $cart  = session()->get('cart');
        $total = 0;
        if($cart){
            foreach($cart as $id=>$item){ $total += $item['price'] * $item['quantity']; }
        }
        $item = CouponCode::where('code',$request->code)
                    ->where(function($row) use($total){
                        $row->where('minimum_amount','0')
                            ->orWhereNull('minimum_amount')
                            ->orWhere('minimum_amount','<=',$total);
                    })
                    ->whereDate('start','<=', date('Y-m-d'))
                    ->whereDate('end','>=', date('Y-m-d'))->first();
        
        if($item){
            session()->put('coupon_discount', $item->amount);
            session()->put('discount_type', $item->discount_type);
            return response()->json(['success'=>true,'msg'=>'You Got Coupon Discount!']);
        }else{
            return response()->json(['success'=>false,'msg'=>'Not Found Any Coupon Discount!']);
        }
    }

    // ✅ FIXED: Notification Logic Added
    private function sendAdminNotification($order) {
        try { 
            $info = Information::first();
            if ($info && $info->notification_active == 1) {
                SendOrderNotification::dispatchAfterResponse($order); 
            }
        } catch (\Exception $e) { 
            \Log::error("Job Dispatch Error: " . $e->getMessage()); 
        }
    }

    public function sendOtp(Request $request) {
        $request->validate([ 'mobile' => 'required|numeric|digits:11' ]);
        if(session()->has('otp_sent_at') && session()->has('otp_mobile')) {
            $lastSent = session()->get('otp_sent_at');
            $lastMobile = session()->get('otp_mobile');
            if($lastMobile == $request->mobile && now()->diffInSeconds($lastSent) < 60) {
                 return response()->json([ 'success' => true, 'msg' => 'আপনার মোবাইলে ৪ ডিজিটের কোড আগেই পাঠানো হয়েছে।' ]);
            }
        }
        $otp = rand(1000, 9999); 
        session()->put('otp_code', $otp);
        session()->put('otp_mobile', $request->mobile);
        session()->put('otp_verified', false);
        session()->put('otp_sent_at', now());
        $msg = "Your OTP code is: " . $otp . " . Please do not share this code.";
        $settings = Information::first();
        if (!$settings || empty($settings->sms_api_key) || empty($settings->sms_sender_id)) {
             return response()->json([ 'success' => false, 'msg' => 'SMS Gateway not configured properly.' ]);
        }
        try {
            $response = Http::get("http://bulksmsbd.net/api/smsapi", [
                'api_key' => $settings->sms_api_key,
                'type' => 'text',
                'number' => $request->mobile,
                'senderid' => $settings->sms_sender_id,
                'message' => $msg,
            ]);
            return response()->json([ 'success' => true, 'msg' => 'আপনার মোবাইলে ৪ ডিজিটের কোড পাঠানো হয়েছে।' ]);
        } catch (\Exception $e) {
            return response()->json([ 'success' => false, 'msg' => 'SMS পাঠাতে সমস্যা হয়েছে: ' . $e->getMessage() ]);
        }
    }

    public function verifyOtp(Request $request) {
        $request->validate([ 'otp' => 'required', 'mobile' => 'required' ]);
        $session_otp = session()->get('otp_code');
        $session_mobile = session()->get('otp_mobile');
        if ($request->mobile == $session_mobile && $request->otp == $session_otp) {
            session()->put('otp_verified', true);
            return response()->json([ 'success' => true, 'msg' => 'ভেরিফিকেশন সফল হয়েছে!' ]);
        } else {
            return response()->json([ 'success' => false, 'msg' => 'ভুল কোড দিয়েছেন! আবার চেষ্টা করুন।' ]);
        }
    }

    // Stock Alert SMS
    private function checkAndSendStockAlert($product)
    {
        if ($product->fresh()->stock_quantity <= 0) {
            $info = Information::first();
            if ($info && $info->sms_api_key && $info->admin_phone) {
                $msg = "Alert: Product '{$product->name}' is now Out of Stock!";
                try {
                    Http::get("http://bulksmsbd.net/api/smsapi", [
                        'api_key' => $info->sms_api_key,
                        'type' => 'text',
                        'number' => $info->admin_phone,
                        'senderid' => $info->sms_sender_id,
                        'message' => $msg,
                    ]);
                } catch (\Exception $e) {
                    \Log::error('Stock Alert SMS Failed: ' . $e->getMessage());
                }
            }
        }
     }
}