<?php

namespace App\Http\Controllers;

use DB;
use Illuminate\Http\Request;
use App\Library\SslCommerz\SslCommerzNotification;
use App\Models\Information;
use App\Models\DeliveryCharge;
use App\Models\User;
use App\Models\Order;
use App\Jobs\SendOrderNotification;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Schema;
use Session;

class SslCommerzPaymentController extends Controller
{
    /**
     * ✅ SSLCommerz Configuration Setup (Dynamic from Database)
     */
    private function setSslConfig()
    {
        $info = Information::first(); 

        $store_id = null;
        $store_pass = null;
        $is_sandbox = true; 

        if ($info) {
            $is_sandbox = ($info->ssl_sandbox == 1); 

            if ($is_sandbox) {
                $store_id = $info->ssl_sandbox_store_id;
                $store_pass = $info->ssl_sandbox_store_password;
            } else {
                $store_id = $info->ssl_store_id;
                $store_pass = $info->ssl_store_password;
            }
        }

        $domain = $is_sandbox ? 'https://sandbox.sslcommerz.com' : 'https://securepay.sslcommerz.com';

        config(['sslcommerz.apiCredentials.store_id' => $store_id]);
        config(['sslcommerz.apiCredentials.store_password' => $store_pass]);
        config(['sslcommerz.apiDomain' => $domain]);
        config(['sslcommerz.connect_from_localhost' => $is_sandbox]);
        config(['sslcommerz.sandbox' => $is_sandbox]);
    }

    /** ===============================
     * ✅ WORKER ASSIGNMENT LOGIC
     * =============================== */
    private function getActiveWorkerIds()
    {
        return User::query()
            ->where(function ($q) {
                $q->whereNull('status')
                    ->orWhereIn('status', [1, '1', true, 'true', 'active', 'Active']);
            })
            ->when(
                Schema::hasColumn((new User)->getTable(), 'deleted_at'),
                fn($q) => $q->whereNull('deleted_at')
            )
            ->whereHas('roles', fn($q) => $q->whereRaw('LOWER(name)=?', ['worker']))
            ->orderBy('id')
            ->pluck('id');
    }

    private function pickNextWorkerId()
    {
        $activeIds = $this->getActiveWorkerIds();
        
        if ($activeIds->isEmpty()) {
            return 1;
        }

        $candidateId = DB::table('users as u')
            ->join('model_has_roles as m', 'm.model_id', '=', 'u.id')
            ->join('roles as r', 'r.id', '=', 'm.role_id')
            ->leftJoin('orders as o', function ($join) {
                $join->on('o.assign_user_id', '=', 'u.id')
                    ->whereDate('o.created_at', DB::raw('CURDATE()'));
            })
            ->whereIn('u.id', $activeIds->toArray())
            ->whereRaw('LOWER(r.name) = ?', ['worker'])
            ->where(function ($q) {
                $q->whereNull('u.status')
                    ->orWhereIn('u.status', [1, '1', true, 'true', 'active', 'Active']);
            })
            ->groupBy('u.id')
            ->orderByRaw('COUNT(o.id) ASC')
            ->orderBy('u.id', 'ASC')
            ->value('u.id');

        return (int) ($candidateId ?? $activeIds->first());
    }

    public function index(Request $request)
    {
        $this->setSslConfig();

        if (empty(config('sslcommerz.apiCredentials.store_id'))) {
            return back()->with('error', 'Payment Gateway Config Missing!');
        }

        if(!$request->amount || !$request->mobile){
            return back()->with('error', 'Amount or Mobile number is missing!');
        }

        // ১. চার্জ এবং ডিসকাউন্ট হিসাব
        $delivery_charge_id = $request->delivery_charge_id;
        $shipping_charge = 0;
        
        if($delivery_charge_id){
            $charge = DeliveryCharge::find($delivery_charge_id);
            if($charge){
                $shipping_charge = $charge->amount;
            }
        }

        $coupon_discount = session()->get('coupon_discount') ?? 0;
        $product_total = $request->amount; 
        
        $final_total_amount = ($product_total + $shipping_charge) - $coupon_discount;

        $post_data = array();
        $post_data['total_amount'] = $final_total_amount;
        $post_data['currency'] = "BDT";
        $post_data['tran_id'] = uniqid(); 

        $post_data['cus_name'] = $request->first_name ?? 'Customer';
        $post_data['cus_email'] = $request->email ?? 'N/A';
        $post_data['cus_add1'] = $request->shipping_address ?? 'Dhaka';
        $post_data['cus_city'] = "";
        $post_data['cus_country'] = "Bangladesh";
        $post_data['cus_phone'] = $request->mobile;

        $post_data['ship_name'] = "Store Test";
        $post_data['ship_add1'] = "Dhaka";
        $post_data['ship_city'] = "Dhaka";
        $post_data['ship_country'] = "Bangladesh";

        $post_data['shipping_method'] = "NO";
        $post_data['product_name'] = "Online Product";
        $post_data['product_category'] = "Goods";
        $post_data['product_profile'] = "physical-goods";

        $assignUserId = $this->pickNextWorkerId();

        DB::beginTransaction();
        try {
            $userId = auth()->id() ?? 0;

            // =========================================================
            // ১. চেক করি এই মোবাইল নাম্বারের কোনো Incomplete অর্ডার আছে কি না
            // =========================================================
            $existingOrder = DB::table('orders')
                ->where('mobile', $post_data['cus_phone'])
                ->where('status', 'incomplete')
                ->orderBy('id', 'desc')
                ->first();

            $orderData = [
                'invoice_no'        => $existingOrder ? $existingOrder->invoice_no : rand(100000, 999999),
                'user_id'           => $userId,
                'first_name'        => $post_data['cus_name'],
                'mobile'            => $post_data['cus_phone'],
                'email'             => $request->email,
                'shipping_address'  => $post_data['cus_add1'],
                'amount'            => $product_total,          
                'discount'          => $coupon_discount,      
                'shipping_charge'   => $shipping_charge,      
                'final_amount'      => $final_total_amount, 
                'status'            => 'Pending', // Incomplete হলে Pending এ আপডেট হবে
                'payment_status'    => 'Unpaid',
                'transaction_id'    => $post_data['tran_id'],
                'currency'          => $post_data['currency'],
                'payment_method'    => 'sslcommerz',
                'assign_user_id'    => $assignUserId, 
                'date'              => date('Y-m-d'),
                'updated_at'        => now(),
            ];

            $order_id = null;

            if ($existingOrder) {
                // যদি Incomplete অর্ডার থাকে, তবে সেটিই আপডেট হবে
                DB::table('orders')->where('id', $existingOrder->id)->update($orderData);
                $order_id = $existingOrder->id;
            } else {
                // না থাকলে নতুন তৈরি হবে
                $orderData['created_at'] = now();
                $order_id = DB::table('orders')->insertGetId($orderData);
            }

            // =========================================================
            // ২. কার্ট আইটেম বা ল্যান্ডিং পেজ আইটেম সেভ (FIXED)
            // =========================================================
            $cart = session()->get('cart', []);
            
            // ✅ যদি ল্যান্ডিং পেজ থেকে সরাসরি অর্ডার আসে (request-এ prd_id থাকলে)
            if ($request->has('prd_id') && !empty($request->prd_id)) {
                
                // আগের অসম্পূর্ণ ডিটেইলস মুছে সর্বশেষ ডেটা ইনসার্ট করা হচ্ছে
                DB::table('order_details')->where('order_id', $order_id)->delete();
                
                $qty = (int) ($request->quantity > 0 ? $request->quantity : 1);
                $unitPrice = $product_total / $qty; 
                
                DB::table('order_details')->insert([
                    'order_id'       => $order_id,
                    'product_id'     => $request->prd_id,
                    'quantity'       => $qty,
                    'unit_price'     => $unitPrice,
                    'purchase_price' => 0, 
                    'discount'       => 0,
                    'variation_id'   => $request->variation_id ?: null, // ✅ ভেরিয়েন্ট সেভ হচ্ছে
                    'created_at'     => now(),
                    'updated_at'     => now(),
                ]);
            } 
            // ✅ যদি সাধারণ কার্ট থেকে অর্ডার আসে
            elseif (!empty($cart)) {
                DB::table('order_details')->where('order_id', $order_id)->delete();

                $orderDetailsData = [];
                foreach ($cart as $item) {
                    $orderDetailsData[] = [
                        'order_id'       => $order_id,
                        'product_id'     => $item['product_id'],
                        'quantity'       => $item['quantity'],
                        'unit_price'     => $item['price'],
                        'purchase_price' => $item['purchase_price'] ?? 0,
                        'discount'       => $item['discount'] ?? 0,
                        'variation_id'   => $item['variation_id'] ?? null,
                        'created_at'     => now(),
                        'updated_at'     => now(),
                    ];
                }
                
                if (!empty($orderDetailsData)) {
                    DB::table('order_details')->insert($orderDetailsData);
                }
            }

            DB::commit(); 

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', "Database Error: " . $e->getMessage());
        }

        $sslc = new SslCommerzNotification();
        $payment_options = $sslc->makePayment($post_data, 'hosted');

        if (!is_array($payment_options)) {
            print_r($payment_options);
            $payment_options = array();
        }
    }

    public function success(Request $request)
    {
        $this->setSslConfig();

        $tran_id = $request->input('tran_id');
        $amount = $request->input('amount');
        $currency = $request->input('currency');

        $sslc = new SslCommerzNotification();

        $order_details = DB::table('orders')->where('transaction_id', $tran_id)->first();

        if (!$order_details) {
            return redirect('/')->with('error', 'Invalid Transaction ID');
        }

        if ($order_details->status == 'Pending') {
            $validation = $sslc->orderValidate($request, $tran_id, $amount, $currency);

            if ($validation) {
                DB::table('orders')->where('transaction_id', $tran_id)->update([
                    'status' => 'Processing',
                    'payment_status' => 'Paid' 
                ]);
                
                // ✅✅✅ NEW: পেমেন্ট সফল হলে ইমেইল ও মেসেজ পাঠানোর আগে সেটিংস চেক
                $order = Order::find($order_details->id);
                $info = Information::first();
                if ($order && $info && $info->notification_active == 1) {
                    SendOrderNotification::dispatch($order);
                }
                // ===============================================
                
                if(session()->has('cart')) session()->forget('cart');
                if(session()->has('coupon_discount')) session()->forget('coupon_discount');

                return redirect()->route('front.confirmOrder', $order_details->id);
            }
        } else if ($order_details->status == 'Processing' || $order_details->status == 'Complete') {
            if(session()->has('cart')) session()->forget('cart');
            return redirect()->route('front.confirmOrder', $order_details->id);
        } else {
            DB::table('orders')->where('transaction_id', $tran_id)->update(['status' => 'Failed']);
            return $this->loadFailedView($order_details);
        }
        
        return $this->loadFailedView($order_details);
    }

    public function fail(Request $request)
    {
        $tran_id = $request->input('tran_id');
        $order_details = DB::table('orders')->where('transaction_id', $tran_id)->first();
        if ($order_details && $order_details->status == 'Pending') {
            DB::table('orders')->where('transaction_id', $tran_id)->update(['status' => 'Failed']);
        }
        return $this->loadFailedView($order_details);
    }

    public function cancel(Request $request)
    {
        $tran_id = $request->input('tran_id');
        $order_details = DB::table('orders')->where('transaction_id', $tran_id)->first();
        if ($order_details && $order_details->status == 'Pending') {
            DB::table('orders')->where('transaction_id', $tran_id)->update(['status' => 'Canceled']);
        }
        return $this->loadFailedView($order_details);
    }

    public function ipn(Request $request)
    {
        $this->setSslConfig();
        
        if ($request->input('tran_id')) {
            $tran_id = $request->input('tran_id');
            $order_details = DB::table('orders')->where('transaction_id', $tran_id)->first();
            
            if ($order_details && $order_details->status == 'Pending') {
                $sslc = new SslCommerzNotification();
                if ($sslc->orderValidate($request, $tran_id, $order_details->amount, $order_details->currency)) {
                    DB::table('orders')->where('transaction_id', $tran_id)->update([
                        'status' => 'Processing',
                        'payment_status' => 'Paid'
                    ]);

                    // ✅✅✅ NEW: IPN থেকে পেমেন্ট সফল হলেও মেসেজ পাঠানোর আগে সেটিংস চেক
                    $order = Order::find($order_details->id);
                    $info = Information::first();
                    if ($order && $info && $info->notification_active == 1) {
                        SendOrderNotification::dispatch($order);
                    }
                    // ===============================================

                    echo "Transaction is successfully Completed";
                }
            }
        }
    }

    private function loadFailedView($order_details = null)
    {
        if (view()->exists('frontend.dashboard.payment_failed')) {
            return view('frontend.dashboard.payment_failed', ['order' => $order_details]);
        }
        return redirect('/')->with('error', 'Payment Failed or Canceled');
    }
}