<?php

namespace App\Services;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Request;
use Illuminate\Support\Facades\Auth;

class FacebookConversionService
{
    private Client $client;
    private $pixelId;
    private $accessToken;

    public function __construct()
    {
        // API Version v19.0 or v20.0 is stable
        $this->client = new Client([
            'base_uri' => 'https://graph.facebook.com/v19.0/',
            'timeout'  => 10.0,
            'verify'   => true,
        ]);

        // Get Credentials (Support for both env and custom setting helper)
        $this->pixelId = $this->getConfig('fb_pixel_id', 'FACEBOOK_PIXEL_ID');
        $this->accessToken = $this->getConfig('fb_access_token', 'FACEBOOK_ACCESS_TOKEN');
    }

    /**
     * Helper to get config from function or env
     */
    private function getConfig($settingKey, $envKey)
    {
        // If you have a global setting() function
        if (function_exists('setting') && setting($settingKey)) {
            return setting($settingKey);
        }
        // Fallback to config/env
        return config("services.facebook.{$settingKey}") ?? env($envKey);
    }

    /* =========================
     * MAIN SENDER
     * ========================= */
    public function sendEvent(
        string $eventName,
        array $customData = [],
        ?string $eventId = null,
        array $override = []
    ): array {
        try {
            if (empty($this->pixelId) || empty($this->accessToken)) {
                // Log only once per request to avoid flooding
                Log::warning('Facebook CAPI skipped: missing ID or Token');
                return ['ok' => false, 'error' => 'Missing Configuration'];
            }

            // 1. Prepare User Data
            $userData = $this->getUserData();

            // Merge override user_data if passed (e.g. from checkout form)
            if (!empty($override['user_data']) && is_array($override['user_data'])) {
                $userData = array_merge($userData, $override['user_data']);
            }

            // Remove null/empty values
            $userData = $this->cleanArray($userData);

            // 2. Prepare Event Data
            $eventData = [
                'event_name'       => $eventName,
                'event_time'       => $override['event_time'] ?? time(),
                'event_source_url' => $override['event_source_url'] ?? request()->fullUrl(),
                'action_source'    => $override['action_source'] ?? 'website',
                'user_data'        => $userData,
            ];

            if (!empty($eventId)) {
                $eventData['event_id'] = $eventId;
            }

            if (!empty($customData)) {
                $eventData['custom_data'] = $customData;
            }

            $payload = ['data' => [$eventData]];

            // Test event code (optional)
            $testCode = $this->getConfig('fb_pixel_test_code', 'FACEBOOK_PIXEL_TEST_CODE');
            if ($testCode) {
                $payload['test_event_code'] = $testCode;
            }

            // 3. Send Request
            $response = $this->client->post($this->pixelId . '/events', [
                'query' => ['access_token' => $this->accessToken],
                'json'  => $payload,
                'headers' => ['Content-Type' => 'application/json'],
            ]);

            $body = json_decode($response->getBody()->getContents(), true);

            return ['ok' => true, 'status' => $response->getStatusCode(), 'body' => $body];

        } catch (\Throwable $e) {
            Log::error('Facebook CAPI Error [' . $eventName . ']: ' . $e->getMessage());
            return ['ok' => false, 'error' => $e->getMessage()];
        }
    }

    /* =========================
     * EVENT METHODS
     * ========================= */

    /**
     * ✅ The Missing Method for CartController
     */
    public function sendAddToCart(array $data, ?string $eventId = null): array
    {
        // Map data structure from Controller to Facebook Custom Data
        $customData = [
            'currency'     => $data['currency'] ?? 'BDT',
            'value'        => (float)($data['value'] ?? 0),
            'content_ids'  => $data['content_ids'] ?? [],
            'content_type' => 'product',
            'contents'     => $data['contents'] ?? [], // [{id, quantity, item_price}]
        ];

        return $this->sendEvent('AddToCart', $customData, $eventId);
    }

    public function sendViewContent(array $data, ?string $eventId = null): array
    {
        $customData = [
            'currency'         => $data['currency'] ?? 'BDT',
            'value'            => (float)($data['value'] ?? 0),
            'content_ids'      => [(string)($data['product_id'] ?? '')],
            'content_name'     => $data['product_name'] ?? '',
            'content_type'     => 'product',
            'content_category' => $data['content_category'] ?? null,
        ];

        return $this->sendEvent('ViewContent', $customData, $eventId);
    }

    public function sendInitiateCheckout(array $data, ?string $eventId = null): array
    {
        $customData = [
            'currency'     => $data['currency'] ?? 'BDT',
            'value'        => (float)($data['value'] ?? 0),
            'num_items'    => (int)($data['num_items'] ?? 1),
            'content_ids'  => $data['content_ids'] ?? [],
            'contents'     => $data['contents'] ?? [],
            'content_type' => 'product',
        ];

        return $this->sendEvent('InitiateCheckout', $customData, $eventId);
    }

    public function sendPurchase(array $data, ?string $eventId = null, array $userData = []): array
    {
        // Purchase usually comes with customer data from the form
        return $this->sendEvent('Purchase', $data, $eventId, [
            'user_data' => $userData
        ]);
    }

    /* =========================
     * USER DATA & HELPERS
     * ========================= */
    private function getUserData(): array
    {
        $request = request();

        $data = [
            'client_ip_address' => $request->ip(),
            'client_user_agent' => $request->userAgent(),
        ];

        // Cookies (Crucial for matching)
        if (!empty($_COOKIE['_fbc'])) $data['fbc'] = $_COOKIE['_fbc'];
        if (!empty($_COOKIE['_fbp'])) $data['fbp'] = $_COOKIE['_fbp'];

        // Authenticated User Data
        $user = Auth::user();
        if ($user) {
            if (!empty($user->email)) {
                $data['em'] = [$this->hash($user->email)];
            }
            if (!empty($user->phone) || !empty($user->phone_number)) {
                $phone = $user->phone ?? $user->phone_number;
                $data['ph'] = [$this->hashPhone($phone)];
            }
            if (!empty($user->name)) {
                // simple split for fn/ln (optional)
                $parts = explode(' ', $user->name, 2);
                $data['fn'] = [$this->hash($parts[0])];
                if (isset($parts[1])) {
                    $data['ln'] = [$this->hash($parts[1])];
                }
            }
        }

        return $data;
    }

    public function hash(?string $value): ?string
    {
        $value = trim(strtolower((string)$value));
        if ($value === '') return null;
        return hash('sha256', $value);
    }

    public function hashPhone(?string $value): ?string
    {
        // Keep only numbers
        $value = preg_replace('/\D+/', '', (string)$value);
        if ($value === '') return null;
        return hash('sha256', $value);
    }

    private function cleanArray(array $arr): array
    {
        $out = [];
        foreach ($arr as $k => $v) {
            if (is_array($v)) {
                $v = array_values(array_filter($v, fn($x) => $x !== null && $x !== ''));
                if (!empty($v)) $out[$k] = $v;
            } else {
                if ($v !== null && $v !== '') $out[$k] = $v;
            }
        }
        return $out;
    }
}